import {
  type Address,
  getProgramDerivedAddress,
  getAddressEncoder,
  getUtf8Encoder,
} from '@solana/kit'
import {
  ACCESS_CONTROL_PREFIX,
  WALLET_ROLE_PREFIX,
  ACCESS_CONTROL_PROGRAM_ADDRESS,
} from './constants'

/**
 * Helper class for deriving Access Control program PDAs
 */
export class AccessControlHelper {
  private mintAddress: Address
  private programId: Address
  public accessControlPubkey: Address | null = null

  constructor(
    mintAddress: Address,
    programId: Address = ACCESS_CONTROL_PROGRAM_ADDRESS,
  ) {
    this.mintAddress = mintAddress
    this.programId = programId
  }

  /**
   * Initialize async properties (call after constructor)
   */
  async init(): Promise<this> {
    this.accessControlPubkey = await this.accessControlPDA()
    return this
  }

  /**
   * Derive wallet role PDA
   */
  async walletRolePDA(walletAddress: Address): Promise<Address> {
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(WALLET_ROLE_PREFIX),
        addressEncoder.encode(this.mintAddress),
        addressEncoder.encode(walletAddress),
      ],
    })
    return pda
  }

  /**
   * Derive access control PDA
   */
  async accessControlPDA(): Promise<Address> {
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(ACCESS_CONTROL_PREFIX),
        addressEncoder.encode(this.mintAddress),
      ],
    })
    return pda
  }
}
