import {
  getCreateAssociatedTokenIdempotentInstruction,
  findAssociatedTokenPda,
  fetchToken,
} from '@solana-program/token-2022'
import { fetchMint as fetchMint2022 } from '@solana-program/token-2022'
import {
  type Address,
  type TransactionSigner,
  createSolanaRpc,
  fetchEncodedAccount,
} from '@solana/kit'
import { TOKEN_2022_PROGRAM_ADDRESS } from './constants'

/**
 * Helper class for Token-2022 mint operations
 */
export class MintHelper {
  private rpc: ReturnType<typeof createSolanaRpc>
  private mintAddress: Address
  private programAddress: Address

  constructor(rpc: ReturnType<typeof createSolanaRpc>, mintAddress: Address) {
    this.rpc = rpc
    this.mintAddress = mintAddress
    this.programAddress = TOKEN_2022_PROGRAM_ADDRESS
  }

  /**
   * Fetch mint account data
   */
  async getMint() {
    return fetchMint2022(this.rpc, this.mintAddress)
  }

  /**
   * Get associated token address for a wallet
   */
  async getAssociatedTokenAddress(walletAddress: Address): Promise<Address> {
    const [ata] = await findAssociatedTokenPda({
      mint: this.mintAddress,
      owner: walletAddress,
      tokenProgram: this.programAddress,
    })
    return ata
  }

  /**
   * Fetch token account data
   */
  async getAccount(tokenAccountAddress: Address) {
    return fetchToken(this.rpc, tokenAccountAddress)
  }

  /**
   * Check if an account exists
   */
  async accountExists(accountAddress: Address): Promise<boolean> {
    try {
      const maybeAccount = await fetchEncodedAccount(this.rpc, accountAddress)
      return maybeAccount.exists
    } catch {
      return false
    }
  }

  /**
   * Create instruction to create associated token account (idempotent)
   */
  createAssociatedTokenAccountInstruction(
    payerSigner: TransactionSigner,
    userWalletAddress: Address,
    userWalletAssociatedTokenAccountAddress: Address,
  ) {
    return getCreateAssociatedTokenIdempotentInstruction({
      payer: payerSigner,
      owner: userWalletAddress,
      mint: this.mintAddress,
      ata: userWalletAssociatedTokenAccountAddress,
      tokenProgram: this.programAddress,
    })
  }
}