import { getSetComputeUnitLimitInstruction } from '@solana-program/compute-budget'
// @ts-ignore: relies on workspace dependency installed at root
import {
  AccountRole,
} from '@solana/instructions'
// @ts-ignore: relies on workspace dependency installed at root
import {
  address,
  appendTransactionMessageInstructions,
  compileTransaction,
  createNoopSigner,
  createSolanaRpc,
  createTransactionMessage,
  pipe,
  setTransactionMessageFeePayerSigner,
  setTransactionMessageLifetimeUsingBlockhash,
  type Address,
  type Instruction,
} from '@solana/kit'
import {
  Connection,
  PublicKey,
  TransactionInstruction,
  VersionedMessage,
  VersionedTransaction,
} from '@solana/web3.js'
import nacl from 'tweetnacl'

import { MintHelper } from './mint-helper'

export function requireEnv(name: string): string {
  const val = process.env[name]
  if (!val || val.includes('REPLACE')) {
    throw new Error(`Set ${name} before running`)
  }
  return val
}

export function parseBoolean(value: string): boolean {
  const normalized = value.trim().toLowerCase()
  if (['true', '1', 'yes'].includes(normalized)) return true
  if (['false', '0', 'no'].includes(normalized)) return false
  throw new Error(`Invalid boolean value: ${value}`)
}

export function makeRpc(rpcUrl: string): {
  connection: Connection
  rpc: ReturnType<typeof createSolanaRpc>
} {
  return {
    connection: new Connection(rpcUrl, 'confirmed'),
    rpc: createSolanaRpc(rpcUrl),
  }
}

export function createPayerSigner(payerAddress: string) {
  const payerAddr = address(payerAddress)
  const payerSigner = createNoopSigner(payerAddr)
  return { payerAddr, payerSigner }
}

export async function ensureAtaForOwner(params: {
  mintHelper: MintHelper
  owner: Address
  payerSigner: ReturnType<typeof createNoopSigner>
  instructions: Instruction[]
}) {
  const { mintHelper, owner, payerSigner, instructions } = params
  const ata = await mintHelper.getAssociatedTokenAddress(owner)
  const exists = await mintHelper.accountExists(ata)
  if (!exists) {
    instructions.push(mintHelper.createAssociatedTokenAccountInstruction(payerSigner, owner, ata))
  }
  return ata
}

export function addComputeUnitLimit(instructions: Instruction[], units: number) {
  instructions.unshift(getSetComputeUnitLimitInstruction({ units }) as unknown as Instruction)
}

export function buildTransactionMessage(params: {
  blockhash: string
  lastValidBlockHeight: number
  feePayer: ReturnType<typeof createNoopSigner>
  instructions: Instruction[]
}): ReturnType<typeof appendTransactionMessageInstructions> {
  const { blockhash, lastValidBlockHeight, feePayer, instructions } = params
  return pipe(
    createTransactionMessage({ version: 0 }),
    tx => setTransactionMessageLifetimeUsingBlockhash({ blockhash, lastValidBlockHeight } as any, tx),
    tx => setTransactionMessageFeePayerSigner(feePayer, tx),
    tx => appendTransactionMessageInstructions(instructions, tx),
  )
}

export function compileAndSignTransaction(params: {
  transactionMessage: ReturnType<typeof appendTransactionMessageInstructions>
  authorityPublicKey: PublicKey
  authoritySecretKey: Uint8Array
}) {
  const { transactionMessage, authorityPublicKey, authoritySecretKey } = params
  const compiledTransaction = compileTransaction(transactionMessage)
  const messageBytes = new Uint8Array(compiledTransaction.messageBytes)
  const tx = new VersionedTransaction(VersionedMessage.deserialize(messageBytes))

  const numRequiredSigners = tx.message.header.numRequiredSignatures
  const signerKeys = tx.message.staticAccountKeys.slice(0, numRequiredSigners)
  const signatures: Buffer[] = Array.from({ length: numRequiredSigners }, () => Buffer.alloc(64))
  const authorityIndex = signerKeys.findIndex(key => key.equals(authorityPublicKey))
  if (authorityIndex === -1) {
    throw new Error('Authority is not a required signer for this transaction')
  }

  signatures[authorityIndex] = Buffer.from(nacl.sign.detached(messageBytes, authoritySecretKey))
  tx.signatures = signatures

  const txBase64 = Buffer.from(tx.serialize()).toString('base64')
  return { tx, txBase64, messageBytes }
}

export function toKitInstruction(ix: TransactionInstruction): Instruction {
  return {
    programAddress: address(ix.programId.toBase58()),
    accounts: ix.keys.map(key => ({
      address: address(key.pubkey.toBase58()),
      role: key.isSigner
        ? key.isWritable
          ? AccountRole.WRITABLE_SIGNER
          : AccountRole.READONLY_SIGNER
        : key.isWritable
          ? AccountRole.WRITABLE
          : AccountRole.READONLY,
    })),
    data: new Uint8Array(ix.data),
  }
}

export function mapToKitInstructions(instructions: TransactionInstruction[]): Instruction[] {
  return instructions.map(toKitInstruction)
}
