import {
  type Address,
  getProgramDerivedAddress,
  getAddressEncoder,
  getUtf8Encoder,
  getU64Encoder,
  createSolanaRpc,
} from '@solana/kit'
import { transferRestrictions } from '@upsideos/solana-rwa'
import {
  TRANSFER_RESTRICTION_DATA_PREFIX,
  TRANSFER_RESTRICTION_GROUP_PREFIX,
  TRANSFER_RESTRICTION_HOLDER_PREFIX,
  TRANSFER_RESTRICTION_HOLDER_GROUP_PREFIX,
  SECURITY_ASSOCIATED_ACCOUNT_PREFIX,
  TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS,
} from './constants'

const { fetchTransferRestrictionData, fetchMaybeSecurityAssociatedAccount } =
  transferRestrictions

type TransferRestrictionData = transferRestrictions.TransferRestrictionData
type SecurityAssociatedAccount = transferRestrictions.SecurityAssociatedAccount

/**
 * Helper class for deriving Transfer Restrictions program PDAs
 */
export class TransferRestrictionsHelper {
  private rpc: ReturnType<typeof createSolanaRpc>
  private mintAddress: Address
  private programId: Address
  public transferRestrictionDataPubkey: Address | null = null

  constructor(
    rpc: ReturnType<typeof createSolanaRpc>,
    mintAddress: Address,
    programId: Address = TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS,
  ) {
    this.rpc = rpc
    this.mintAddress = mintAddress
    this.programId = programId
  }

  /**
   * Initialize async properties (call after constructor)
   */
  async init(): Promise<this> {
    this.transferRestrictionDataPubkey = await this.transferRestrictionDataPDA()
    return this
  }

  /**
   * Derive transfer restriction data PDA
   */
  async transferRestrictionDataPDA(): Promise<Address> {
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(TRANSFER_RESTRICTION_DATA_PREFIX),
        addressEncoder.encode(this.mintAddress),
      ],
    })
    return pda
  }

  /**
   * Fetch transfer restriction data
   */
  async transferRestrictionData(): Promise<TransferRestrictionData> {
    if (!this.transferRestrictionDataPubkey) {
      throw new Error('Helper not initialized. Call init() first.')
    }
    const account = await fetchTransferRestrictionData(
      this.rpc,
      this.transferRestrictionDataPubkey,
    )
    return account.data
  }

  /**
   * Derive group PDA
   */
  async groupPDA(groupId: bigint): Promise<Address> {
    if (!this.transferRestrictionDataPubkey) {
      throw new Error('Helper not initialized. Call init() first.')
    }
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()
    const u64Encoder = getU64Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(TRANSFER_RESTRICTION_GROUP_PREFIX),
        addressEncoder.encode(this.transferRestrictionDataPubkey),
        u64Encoder.encode(groupId),
      ],
    })
    return pda
  }

  /**
   * Derive holder PDA
   */
  async holderPDA(holderId: bigint): Promise<Address> {
    if (!this.transferRestrictionDataPubkey) {
      throw new Error('Helper not initialized. Call init() first.')
    }
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()
    const u64Encoder = getU64Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(TRANSFER_RESTRICTION_HOLDER_PREFIX),
        addressEncoder.encode(this.transferRestrictionDataPubkey),
        u64Encoder.encode(holderId),
      ],
    })
    return pda
  }

  /**
   * Derive holder group PDA
   */
  async holderGroupPDA(
    holderPubkey: Address,
    groupId: bigint,
  ): Promise<Address> {
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()
    const u64Encoder = getU64Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(TRANSFER_RESTRICTION_HOLDER_GROUP_PREFIX),
        addressEncoder.encode(holderPubkey),
        u64Encoder.encode(groupId),
      ],
    })
    return pda
  }

  /**
   * Derive security associated account PDA
   */
  async securityAssociatedAccountPDA(
    userWalletAssociatedAccountPubkey: Address,
  ): Promise<Address> {
    const addressEncoder = getAddressEncoder()
    const utf8Encoder = getUtf8Encoder()

    const [pda] = await getProgramDerivedAddress({
      programAddress: this.programId,
      seeds: [
        utf8Encoder.encode(SECURITY_ASSOCIATED_ACCOUNT_PREFIX),
        addressEncoder.encode(userWalletAssociatedAccountPubkey),
      ],
    })
    return pda
  }

  /**
   * Fetch security associated account data
   * Returns null if account doesn't exist
   */
  async securityAssociatedAccountData(
    securityAssociatedAccountPubkey: Address,
  ): Promise<SecurityAssociatedAccount | null> {
    const maybeAccount = await fetchMaybeSecurityAssociatedAccount(
      this.rpc,
      securityAssociatedAccountPubkey,
    )

    if (!maybeAccount.exists) {
      return null
    }

    return maybeAccount.data
  }
}
