#!/usr/bin/env -S npx tsx
/**
 * Minimal mintSecurities builder (authority-signed, base64 only).
 */

import { accessControl } from '@upsideos/solana-rwa'
import { Keypair, clusterApiUrl } from '@solana/web3.js'
import { address, createNoopSigner, type Instruction } from '@solana/kit'

import { ACCESS_CONTROL_PROGRAM_ADDRESS } from './helpers/constants'
import { MintHelper } from './helpers/mint-helper'
import { TransferRestrictionsHelper } from './helpers/transfer-restriction-helper'
import {
  addComputeUnitLimit,
  buildTransactionMessage,
  compileAndSignTransaction,
  createPayerSigner,
  ensureAtaForOwner,
  makeRpc,
  requireEnv,
} from './helpers/transaction-utils'

const { getMintSecuritiesInstructionAsync } = accessControl

const RPC_URL = process.env.RPC_URL || clusterApiUrl('devnet')
const TOKEN_MINT = requireEnv('TOKEN_MINT')
const DESTINATION_WALLET = requireEnv('DESTINATION_WALLET')
const PAYER_ADDRESS = process.env.RELAYER_FEE_PAYER || requireEnv('PAYER_ADDRESS')
const AMOUNT_RAW = BigInt(requireEnv('AMOUNT_RAW'))
const AUTHORITY_PRIVATE_KEY_B64 = requireEnv('AUTHORITY_PRIVATE_KEY_B64')

async function main(): Promise<void> {
  const { connection, rpc } = makeRpc(RPC_URL)
  const authority = Keypair.fromSecretKey(Buffer.from(AUTHORITY_PRIVATE_KEY_B64, 'base64'))
  const { payerAddr, payerSigner } = createPayerSigner(PAYER_ADDRESS)
  const mint = address(TOKEN_MINT)
  const walletAddr = address(DESTINATION_WALLET)

  const mintHelper = new MintHelper(rpc, mint)
  const instructions: Instruction[] = []

  const destAta = await ensureAtaForOwner({ mintHelper, owner: walletAddr, payerSigner, instructions })

  const transferRestrictionsHelper = new TransferRestrictionsHelper(rpc, mint)


  const securityAssociatedAccount = await transferRestrictionsHelper.securityAssociatedAccountPDA(destAta)

  const mintIx = await getMintSecuritiesInstructionAsync(
    {
      // Pass a signer so the compiled message requires the authority signature
      authority: createNoopSigner(address(authority.publicKey.toString())) as any,
      securityMint: mint,
      destinationAuthority: walletAddr,
      destinationAccount: destAta,
      securityAssociatedAccount: securityAssociatedAccount,
      amount: AMOUNT_RAW,
    },
    { programAddress: ACCESS_CONTROL_PROGRAM_ADDRESS },
  )

  addComputeUnitLimit(instructions, 300_000)
  instructions.push(mintIx as Instruction)

  const { blockhash, lastValidBlockHeight } = await connection.getLatestBlockhash()

  const transactionMessage = buildTransactionMessage({
    blockhash,
    lastValidBlockHeight,
    feePayer: payerSigner,
    instructions,
  })

  const { txBase64 } = compileAndSignTransaction({
    transactionMessage,
    authorityPublicKey: authority.publicKey,
    authoritySecretKey: authority.secretKey,
  })

  console.log('Authority address:', authority.publicKey.toString())
  console.log('Payer (relayer):', payerAddr.toString())
  console.log('Destination ATA:', destAta.toString())
  console.log('SecurityAssociatedAccount:', securityAssociatedAccount)
  console.log('Last valid block height:', lastValidBlockHeight)
  console.log('txBase64:', txBase64)
}

main().catch(err => {
  console.error('❌ Failed:', err)
  process.exit(1)
})


