#!/usr/bin/env -S npx tsx
/**
 * Minimal setAddressPermissions builder (authority-signed, base64 only).
 *
 * Builds the transaction with the provided authority key, leaving the payer
 * signature empty so it can be signed by the relayer.
 */

import { transferRestrictions } from '@upsideos/solana-rwa'
import { Keypair, clusterApiUrl } from '@solana/web3.js'
// @ts-ignore: relies on workspace dependency installed at root
import {
  address,
  createNoopSigner,
  createSolanaRpc,
  fetchEncodedAccount,
  type Address,
  type Instruction,
} from '@solana/kit'

import { ACCESS_CONTROL_PROGRAM_ADDRESS, TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS } from './helpers/constants'
import { AccessControlHelper } from './helpers/access-control-helper'
import { MintHelper } from './helpers/mint-helper'
import { TransferRestrictionsHelper } from './helpers/transfer-restriction-helper'
import { AccountState } from '@solana-program/token-2022'
import {
  addComputeUnitLimit,
  buildTransactionMessage,
  compileAndSignTransaction,
  createPayerSigner,
  ensureAtaForOwner,
  makeRpc,
  parseBoolean,
  requireEnv,
} from './helpers/transaction-utils'

const {
  getSetAddressPermissionInstructionAsync,
  getInitializeTransferRestrictionHolderInstructionAsync,
  getInitializeHolderGroupInstruction,
} = transferRestrictions

const RPC_URL = process.env.RPC_URL || clusterApiUrl('devnet')
const TOKEN_MINT = requireEnv('TOKEN_MINT')
const WALLET_ADDRESS = requireEnv('WALLET_ADDRESS')
const GROUP_ID = BigInt(requireEnv('GROUP_ID'))
const FROZEN = parseBoolean(requireEnv('FROZEN'))
const PAYER_ADDRESS = process.env.RELAYER_FEE_PAYER || requireEnv('PAYER_ADDRESS')
const AUTHORITY_PRIVATE_KEY_B64 = requireEnv('AUTHORITY_PRIVATE_KEY_B64')

async function accountExists(rpc: ReturnType<typeof createSolanaRpc>, accountAddress: Address): Promise<boolean> {
  try {
    const maybeAccount = await fetchEncodedAccount(rpc, accountAddress)
    return maybeAccount.exists
  } catch {
    return false
  }
}

async function main(): Promise<void> {
  const { connection, rpc } = makeRpc(RPC_URL)

  const authority = Keypair.fromSecretKey(Buffer.from(AUTHORITY_PRIVATE_KEY_B64, 'base64'))
  const authorityAddr = address(authority.publicKey.toBase58())
  const authoritySigner = createNoopSigner(authorityAddr)
  const { payerAddr, payerSigner } = createPayerSigner(PAYER_ADDRESS)
  const mintAddr = address(TOKEN_MINT)
  const walletAddr = address(WALLET_ADDRESS)

  console.log('🔑 Authority:', authorityAddr.toString())
  console.log('💸 Payer (relayer):', payerAddr.toString())
  console.log('🏷️ Wallet:', walletAddr.toString())
  console.log('🪙 Token mint:', mintAddr.toString())
  console.log('👥 Group ID:', GROUP_ID.toString())
  console.log('🧊 Frozen:', FROZEN)
  console.log('🌐 RPC:', RPC_URL)

  const mintHelper = new MintHelper(rpc, mintAddr)
  const accessControlHelper = await new AccessControlHelper(mintAddr, ACCESS_CONTROL_PROGRAM_ADDRESS).init()
  const transferRestrictionsHelper = await new TransferRestrictionsHelper(
    rpc,
    mintAddr,
    TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS,
  ).init()

  const instructions: Instruction[] = []

  const walletAta = await ensureAtaForOwner({ mintHelper, owner: walletAddr, payerSigner, instructions })

  const authorityWalletRole = await accessControlHelper.walletRolePDA(authorityAddr)
  const securityAssociatedAccount = await transferRestrictionsHelper.securityAssociatedAccountPDA(walletAta)
  const secAssocAccountData = await transferRestrictionsHelper.securityAssociatedAccountData(securityAssociatedAccount)
  const hasHolder =
    secAssocAccountData !== null &&
    secAssocAccountData.holder !== null &&
    secAssocAccountData.holder.__option === 'Some'

  if (!hasHolder) {
    await buildInstructionsForNewAccount({
      groupId: GROUP_ID,
      frozen: FROZEN,
      instructions,
      accessControlHelper,
      transferRestrictionsHelper,
      authorityWalletRole,
      securityAssociatedAccount,
      walletAddr,
      walletAta,
      mintAddr,
      authoritySigner,
      payerSigner,
      rpc,
    })
  } else {
    await buildInstructionsForExistingAccount({
      groupId: GROUP_ID,
      frozen: FROZEN,
      instructions,
      mintHelper,
      accessControlHelper,
      transferRestrictionsHelper,
      authorityWalletRole,
      securityAssociatedAccount,
      walletAddr,
      walletAta,
      secAssocAccountData: secAssocAccountData!,
      mintAddr,
      authoritySigner,
      payerSigner,
      rpc,
    })
  }

  // Raise CU limit for complex flows
  addComputeUnitLimit(instructions, 400_000)

  const { blockhash, lastValidBlockHeight } = await connection.getLatestBlockhash()

  const transactionMessage = buildTransactionMessage({
    blockhash,
    lastValidBlockHeight,
    feePayer: payerSigner,
    instructions,
  })

  const { txBase64 } = compileAndSignTransaction({
    transactionMessage,
    authorityPublicKey: authority.publicKey,
    authoritySecretKey: authority.secretKey,
  })

  console.log('SecurityAssociatedAccount:', securityAssociatedAccount.toString())
  console.log('Wallet ATA:', walletAta.toString())
  console.log('Last valid block height:', lastValidBlockHeight)
  console.log('txBase64:', txBase64)
}

type NewAccountParams = {
  groupId: bigint
  frozen: boolean
  instructions: Instruction[]
  accessControlHelper: AccessControlHelper
  transferRestrictionsHelper: TransferRestrictionsHelper
  authorityWalletRole: Address
  securityAssociatedAccount: Address
  walletAddr: Address
  walletAta: Address
  mintAddr: Address
  authoritySigner: ReturnType<typeof createNoopSigner>
  payerSigner: ReturnType<typeof createNoopSigner>
  rpc: ReturnType<typeof createSolanaRpc>
}

async function buildInstructionsForNewAccount({
  groupId,
  frozen,
  instructions,
  accessControlHelper,
  transferRestrictionsHelper,
  authorityWalletRole,
  securityAssociatedAccount,
  walletAddr,
  walletAta,
  mintAddr,
  authoritySigner,
  payerSigner,
  rpc,
}: NewAccountParams) {
  console.log('🆕 Building instructions for NEW account')

  const transferRestrictionDataPubkey = transferRestrictionsHelper.transferRestrictionDataPubkey
  if (!transferRestrictionDataPubkey) {
    throw new Error('Transfer restriction data PDA missing. Call init() first.')
  }
  const initializedTransferRestrictionDataPubkey: Address = transferRestrictionDataPubkey
  const accessControlPubkey = accessControlHelper.accessControlPubkey
  if (!accessControlPubkey) {
    throw new Error('Access control PDA missing. Call init() first.')
  }

  const transferRestrictionData = await transferRestrictionsHelper.transferRestrictionData()
  const holderId = BigInt(transferRestrictionData.holderIds)
  const holderAddress = await transferRestrictionsHelper.holderPDA(holderId)

  const holderExists = await accountExists(rpc, holderAddress)
  if (!holderExists) {
    const initHolderIx = await getInitializeTransferRestrictionHolderInstructionAsync(
      {
        transferRestrictionData: initializedTransferRestrictionDataPubkey,
        accessControlAccount: accessControlPubkey,
        authorityWalletRole,
        authority: authoritySigner,
        payer: payerSigner,
        id: holderId,
      },
      { programAddress: TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS },
    )
    instructions.push(initHolderIx)
  }

  const groupNewAddress = await transferRestrictionsHelper.groupPDA(groupId)
  const holderGroupNewAddress = await transferRestrictionsHelper.holderGroupPDA(holderAddress, groupId)

  const groupExists = await accountExists(rpc, groupNewAddress)
  if (!groupExists) {
    throw new Error(`Transfer Group ${groupId.toString()} not found`)
  }

  const holderGroupExists = await accountExists(rpc, holderGroupNewAddress)
  if (!holderGroupExists) {
    const initHolderGroupIx = getInitializeHolderGroupInstruction(
      {
        holderGroup: holderGroupNewAddress,
        transferRestrictionData: initializedTransferRestrictionDataPubkey,
        group: groupNewAddress,
        holder: holderAddress,
        authorityWalletRole,
        authority: authoritySigner,
        payer: payerSigner,
      },
      { programAddress: TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS },
    )
    instructions.push(initHolderGroupIx)
  }

  const setAddressPermissionIx = await getSetAddressPermissionInstructionAsync(
    {
      securityAssociatedAccount,
      transferRestrictionGroupNew: groupNewAddress,
      transferRestrictionGroupCurrent: undefined,
      transferRestrictionHolder: holderAddress,
      holderGroupNew: holderGroupNewAddress,
      holderGroupCurrent: undefined,
      securityToken: mintAddr,
      userWallet: walletAddr,
      userAssociatedTokenAccount: walletAta,
      authorityWalletRole,
      securityMint: mintAddr,
      accessControlAccount: accessControlHelper.accessControlPubkey!,
      accessControlProgram: ACCESS_CONTROL_PROGRAM_ADDRESS,
      payer: payerSigner,
      authority: authoritySigner,
      groupId,
      frozen,
    },
    { programAddress: TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS },
  )
  instructions.push(setAddressPermissionIx as Instruction)
}

type ExistingAccountParams = {
  groupId: bigint
  frozen: boolean
  instructions: Instruction[]
  mintHelper: MintHelper
  accessControlHelper: AccessControlHelper
  transferRestrictionsHelper: TransferRestrictionsHelper
  authorityWalletRole: Address
  securityAssociatedAccount: Address
  walletAddr: Address
  walletAta: Address
  secAssocAccountData: NonNullable<
    Awaited<ReturnType<TransferRestrictionsHelper['securityAssociatedAccountData']>>
  >
  mintAddr: Address
  authoritySigner: ReturnType<typeof createNoopSigner>
  payerSigner: ReturnType<typeof createNoopSigner>
  rpc: ReturnType<typeof createSolanaRpc>
}

async function buildInstructionsForExistingAccount({
  groupId,
  frozen,
  instructions,
  mintHelper,
  accessControlHelper,
  transferRestrictionsHelper,
  authorityWalletRole,
  securityAssociatedAccount,
  walletAddr,
  walletAta,
  secAssocAccountData,
  mintAddr,
  authoritySigner,
  payerSigner,
  rpc,
}: ExistingAccountParams) {
  console.log('♻️ Building instructions for EXISTING account')

  const currentGroupId = Number(secAssocAccountData.group)
  if (secAssocAccountData.holder.__option !== 'Some') {
    throw new Error('Security associated account holder not found')
  }
  const holderAddress = secAssocAccountData.holder.value

  const tokenAccount = await mintHelper.getAccount(walletAta)
  const currentlyFrozen = tokenAccount.data.state === AccountState.Frozen

  const shouldUpdateGroup = currentGroupId !== Number(groupId)
  const shouldUpdateFrozen = currentlyFrozen !== frozen
  if (!shouldUpdateGroup && !shouldUpdateFrozen) {
    throw new Error('Wallet already has the specified group and frozen state')
  }

  const groupCurrentAddress = await transferRestrictionsHelper.groupPDA(BigInt(currentGroupId))
  const groupNewAddress = await transferRestrictionsHelper.groupPDA(groupId)
  const holderGroupCurrentAddress = await transferRestrictionsHelper.holderGroupPDA(
    holderAddress,
    BigInt(currentGroupId),
  )
  const holderGroupNewAddress = await transferRestrictionsHelper.holderGroupPDA(holderAddress, groupId)

  const groupExists = await accountExists(rpc, groupNewAddress)
  if (!groupExists) {
    throw new Error(`Transfer Group ${groupId.toString()} not found`)
  }

  const holderGroupExists = await accountExists(rpc, holderGroupNewAddress)
  if (!holderGroupExists) {
    const initHolderGroupIx = getInitializeHolderGroupInstruction(
      {
        holderGroup: holderGroupNewAddress,
        transferRestrictionData: transferRestrictionsHelper.transferRestrictionDataPubkey!,
        group: groupNewAddress,
        holder: holderAddress,
        authorityWalletRole,
        authority: authoritySigner,
        payer: payerSigner,
      },
      { programAddress: TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS },
    )
    instructions.push(initHolderGroupIx)
  }

  const setAddressPermissionIx = await getSetAddressPermissionInstructionAsync(
    {
      securityAssociatedAccount,
      transferRestrictionGroupNew: groupNewAddress,
      transferRestrictionGroupCurrent: groupCurrentAddress,
      transferRestrictionHolder: holderAddress,
      holderGroupNew: holderGroupNewAddress,
      holderGroupCurrent: holderGroupCurrentAddress,
      securityToken: mintAddr,
      userWallet: walletAddr,
      userAssociatedTokenAccount: walletAta,
      authorityWalletRole,
      securityMint: mintAddr,
      accessControlAccount: accessControlHelper.accessControlPubkey!,
      accessControlProgram: ACCESS_CONTROL_PROGRAM_ADDRESS,
      payer: payerSigner,
      authority: authoritySigner,
      groupId,
      frozen,
    },
    { programAddress: TRANSFER_RESTRICTIONS_PROGRAM_ADDRESS },
  )
  instructions.push(setAddressPermissionIx as Instruction)
}

main().catch(err => {
  console.error('❌ Failed:', err)
  process.exit(1)
})
