#!/usr/bin/env -S npx tsx
/**
 * Minimal transfer-with-hook builder (authority-signed, base64 only).
 */

// @ts-ignore: relies on workspace dependency installed at root
import { AccountRole } from '@solana/instructions'
import {
  createAssociatedTokenAccountInstruction,
  createTransferCheckedWithTransferHookInstruction,
  getAssociatedTokenAddress,
  getMint,
  TOKEN_2022_PROGRAM_ID,
} from '@solana/spl-token'
import { ComputeBudgetProgram, PublicKey, TransactionInstruction, clusterApiUrl } from '@solana/web3.js'
// @ts-ignore: relies on workspace dependency installed at root
import { address } from '@solana/kit'
import nacl from 'tweetnacl'

import {
  addComputeUnitLimit,
  buildTransactionMessage,
  compileAndSignTransaction,
  createPayerSigner,
  makeRpc,
  mapToKitInstructions,
  requireEnv,
} from './helpers/transaction-utils'

const RPC_URL = process.env.RPC_URL || clusterApiUrl('devnet')
const TOKEN_MINT = requireEnv('TOKEN_MINT')
const TO_WALLET = requireEnv('TO_WALLET')
const PAYER_ADDRESS = process.env.RELAYER_FEE_PAYER || requireEnv('PAYER_ADDRESS')
const AMOUNT_RAW = BigInt(requireEnv('AMOUNT_RAW'))
const AUTHORITY_PRIVATE_KEY_B64 = requireEnv('AUTHORITY_PRIVATE_KEY_B64')

async function main(): Promise<void> {
  const { connection } = makeRpc(RPC_URL)
  const authority = nacl.sign.keyPair.fromSecretKey(Buffer.from(AUTHORITY_PRIVATE_KEY_B64, 'base64'))
  const authorityPk = new PublicKey(authority.publicKey)
  const { payerAddr, payerSigner } = createPayerSigner(PAYER_ADDRESS)
  const payerPk = new PublicKey(payerAddr)
  const mintPk = new PublicKey(TOKEN_MINT)
  const toWalletPk = new PublicKey(TO_WALLET)

  console.log('🔑 Authority (from):', authorityPk.toBase58())
  console.log('💸 Payer (relayer):', payerAddr.toString())
  console.log('🎯 Recipient wallet:', TO_WALLET)
  console.log('🪙 Token mint:', TOKEN_MINT)
  console.log('🌐 RPC:', RPC_URL)
  console.log('🔢 Amount (raw units):', AMOUNT_RAW.toString())

  const decimals = (await getMint(connection, mintPk, 'confirmed', TOKEN_2022_PROGRAM_ID)).decimals

  const fromAta = await getAssociatedTokenAddress(mintPk, authorityPk, false, TOKEN_2022_PROGRAM_ID)
  const toAta = await getAssociatedTokenAddress(mintPk, toWalletPk, false, TOKEN_2022_PROGRAM_ID)

  const fromAtaInfo = await connection.getAccountInfo(fromAta)
  if (!fromAtaInfo) {
    throw new Error('Sender associated token account not found')
  }

  const senderBalance = await connection.getTokenAccountBalance(fromAta)
  const senderAmount = BigInt(senderBalance.value.amount)
  if (AMOUNT_RAW > senderAmount) {
    throw new Error(
      `Sender balance too low: have ${senderAmount.toString()}, need ${AMOUNT_RAW.toString()}`,
    )
  }

  const instructions: TransactionInstruction[] = []
  // Set a higher CU limit for transfer-with-hook processing
  instructions.push(ComputeBudgetProgram.setComputeUnitLimit({ units: 300_000 }))

  const toAtaInfo = await connection.getAccountInfo(toAta)
  if (!toAtaInfo) {
    instructions.push(
      createAssociatedTokenAccountInstruction(
        payerPk,
        toAta,
        toWalletPk,
        mintPk,
        TOKEN_2022_PROGRAM_ID,
      ),
    )
  }

  const transferIx = await createTransferCheckedWithTransferHookInstruction(
    connection,
    fromAta,
    mintPk,
    toAta,
    authorityPk,
    AMOUNT_RAW,
    decimals,
    undefined,
    'confirmed',
    TOKEN_2022_PROGRAM_ID,
  )
  instructions.push(transferIx)

  const kitInstructions = mapToKitInstructions(instructions)

  const { blockhash, lastValidBlockHeight } = await connection.getLatestBlockhash()

  const transactionMessage = buildTransactionMessage({
    blockhash,
    lastValidBlockHeight,
    feePayer: payerSigner,
    instructions: kitInstructions,
  })

  const { txBase64 } = compileAndSignTransaction({
    transactionMessage,
    authorityPublicKey: authorityPk,
    authoritySecretKey: authority.secretKey,
  })

  console.log('From ATA:', fromAta.toBase58())
  console.log('To ATA:', toAta.toBase58())
  console.log('Last valid block height:', lastValidBlockHeight)
  console.log('txBase64:', txBase64)
}

main().catch(err => {
  console.error('❌ Failed:', err)
  process.exit(1)
})


